<?php  
class Map  
{  
    private $background;  
    private $mapWidth;  
    private $mapHeight;  
    private $blipFolder = "blips"; // Default blip folder  
    private $blips = [];  
    private $radarAreas = [];  
    private $texts = [];  // Array to hold text data

    // Constructor to load the map image  
    public function __construct($filename)  
    {  
        $this->loadMap($filename);  
    }  

    // Load the background map image  
    private function loadMap($filename)  
    {  
        if (!file_exists($filename)) {
            throw new Exception("Map file not found: $filename");
        }

        // Check the image type  
        $imageType = exif_imagetype($filename);  
        if ($imageType === IMAGETYPE_JPEG) {  
            $this->background = imagecreatefromjpeg($filename);  
        } elseif ($imageType === IMAGETYPE_PNG) {  
            $this->background = imagecreatefrompng($filename);  
        } elseif ($imageType === IMAGETYPE_GIF) {  
            $this->background = imagecreatefromgif($filename);  
        } else {  
            throw new Exception('Error loading map image. Unsupported format.');  
        }  
        $this->mapWidth = imagesx($this->background);  
        $this->mapHeight = imagesy($this->background);  
    }  

    // Set blip folder  
    public function setBlipFolder($folder)  
    {  
        if (!is_dir($folder)) {
            throw new Exception("Blip folder not found: $folder");
        }
        $this->blipFolder = rtrim($folder, '/') . '/';  
    }  

    // Create a blip and add it to the map  
    public function createBlip($blipId, $x, $y)  
    {  
        $this->blips[] = ['blipId' => $blipId, 'x' => $x, 'y' => $y];  
    }  

    // Create a radar area  
    public function createRadarArea($startPosX, $startPosY, $sizeX, $sizeY, $r = 255, $g = 0, $b = 0, $a = 50)  
    {  
        $this->radarAreas[] = [$startPosX, $startPosY, $sizeX, $sizeY, $r, $g, $b, $a];  
    }  

    // Create text  
    public function createText($text, $posX, $posY, $size, $font)  
    {  
        if (!file_exists($font)) {
            throw new Exception("Font file not found: $font");
        }
        $this->texts[] = ['text' => $text, 'x' => $posX, 'y' => $posY, 'size' => $size, 'font' => $font];  
    }  

    // Converts MTA coordinates to image coordinates  
    private function convertCoordinates($x, $y)  
    {  
        $imgX = ($x + 3000) / 6000 * $this->mapWidth; // Scale to image width  
        $imgY = $this->mapHeight - (($y + 3000) / 6000 * $this->mapHeight); // Invert Y and scale to image height  
        return [$imgX, $imgY];  
    }  

    // Draw radar areas on the map  
    private function drawRadarAreas()  
    {  
        foreach ($this->radarAreas as $areaData) {  
            list($x, $y, $width, $height, $r, $g, $b, $alpha) = $areaData;  
            list($imgX, $imgY) = $this->convertCoordinates($x, $y);  
            $imgWidth = ($width / 6000) * $this->mapWidth;  
            $imgHeight = ($height / 6000) * $this->mapHeight;  
            $color = imagecolorallocatealpha($this->background, $r, $g, $b, $alpha);  
            imagefilledrectangle($this->background, $imgX, $imgY, $imgX + $imgWidth, $imgY + $imgHeight, $color);  
        }  
    }


    // Draw blips on the map  
    private function drawBlips()  
    {  
        foreach ($this->blips as $blipData) {  
            list($blipId, $x, $y) = [$blipData['blipId'], $blipData['x'], $blipData['y']];  
            list($imgX, $imgY) = $this->convertCoordinates($x, $y);  
            // Load the blip image (suppress warnings)  
            $blipImage = @imagecreatefrompng($this->blipFolder . "$blipId.png");  
            if ($blipImage) {  
                // Draw the blip on the map  
                imagecopyresampled($this->background, $blipImage, $imgX - 25, $imgY - 25, 0, 0, 50, 50, imagesx($blipImage), imagesy($blipImage));  
                imagedestroy($blipImage); // Free up memory  
            } else {  
                // Log or handle the error if the image could not be loaded  
                error_log("Error loading blip image: " . $this->blipFolder . "$blipId.png");  
            }  
        }  
    }  

    // Draw text on the map  
    private function drawText()  
    {  
        foreach ($this->texts as $textData) {  
            list($text, $x, $y, $size, $font) = [$textData['text'], $textData['x'], $textData['y'], $textData['size'], $textData['font']];  
            list($imgX, $imgY) = $this->convertCoordinates($x, $y);  
            $color = imagecolorallocate($this->background, 255, 255, 255); // White color for text  
            // Write the text on the image
            imagettftext($this->background, $size, 0, $imgX, $imgY, $color, $font, $text);  
        }  
    }  

    // Output the image  
    public function render($outputFormat = 'png', $filename = 'map.png')  
    {  
        // Set the content type based on output format  
        $contentType = 'image/png';  
        if ($outputFormat === 'jpeg') {  
            $contentType = 'image/jpeg';  
        }  
        header('Content-type: image/png');
        // header("Content-Disposition: inline; filename=\"$filename\"");  
        
        // Draw all elements on the map  
        $this->drawRadarAreas();  
        $this->drawBlips();  
        $this->drawText();  // Draw texts on the map
        
        // Output the image  
        if ($outputFormat === 'jpeg') {  
            imagejpeg($this->background);  
        } else {  
            imagepng($this->background);  
        }  
        imagedestroy($this->background);  
    }  
}
?>
